<?php
/* --------------------------------------------------------------
  JsonTransformer.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit;

use RuntimeException;

/**
 * Class JsonTransformer
 *
 * @package StyleEdit
 */
class JsonTransformer
{
	/**
	 * Encodes the given data and returns the json encoded string.
	 *
	 * @param mixed $p_value Data to be JSON-encoded
	 *
	 * @throws \RuntimeException if JSON encoding failed
	 *
	 * @return string JSON formatted string
	 */
	public static function encode($p_value)
	{
		if(defined('JSON_PRETTY_PRINT') && defined('JSON_UNESCAPED_SLASHES'))
		{
			$json = json_encode($p_value, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
		}
		else
		{
			$json = json_encode($p_value); // PHP v5.3
		}
		
		if(json_last_error())
		{
			$errorMessage = json_last_error_msg();
			throw new RuntimeException("JSON Encode Error: $errorMessage");
		}
		
		return $json;
	}
	
	
	/**
	 * Decode and returns the given json formatted string.
	 *
	 * @param string $p_jsonFormattedString The json formatted string.
	 * @param bool   $p_assoc               When true, returned objects will be converted into associative arrays.
	 *
	 * @throws \RuntimeException if errors occur during the decoding process.
	 *
	 * @return mixed The json data as an array or object depending on the $p_assoc parameter.
	 */
	public static function decode($p_jsonFormattedString, $p_assoc = true)
	{
		$json = json_decode($p_jsonFormattedString, (bool)$p_assoc);
		
		if(json_last_error())
		{
			$errorMessage = json_last_error_msg();
			throw new RuntimeException("JSON Decode Error: $errorMessage");
		}
		
		return $json;
	}
}